// modules/Chat/resources/js/global-chat.js - VERSÃO COMPLETA COM NOTIFICAÇÕES CORRIGIDAS

let globalChatInitialized = false
let websocketService = null
let chatStore = null
let lastNotification = { type: null, time: 0 }
let globalEventListeners = []
let cleanupTimers = []
let sortTimeout = null

window.chatGlobalInitialized = false

console.log('✅ Chat Global: Módulo carregado')

// ===== CONFIGURAÇÕES =====
const notificationConfig = {
    enabled: true,
    sound: true,
    desktop: true,
    maxVisible: 3,
    duration: 5000,
    throttleTime: 3000
}

const metrics = {
    messagesReceived: 0,
    notificationsShown: 0,
    reconnections: 0,
    errors: 0,
    startTime: Date.now()
}

// ===== CLEANUP FUNCTION =====
function cleanupGlobalChat() {
    console.log('🧹 Chat Global: Limpando recursos...')
    
    // Limpar event listeners
    globalEventListeners.forEach(({ element, event, handler }) => {
        try {
            element.removeEventListener(event, handler)
        } catch (error) {
            console.warn('⚠️ Erro ao remover listener:', error)
        }
    })
    globalEventListeners = []
    
    // Limpar timers
    cleanupTimers.forEach(timerId => {
        try {
            clearTimeout(timerId)
        } catch (error) {
            console.warn('⚠️ Erro ao limpar timer:', error)
        }
    })
    cleanupTimers = []
    
    // Limpar sort timeout
    if (sortTimeout) {
        clearTimeout(sortTimeout)
        sortTimeout = null
    }
    
    // Limpar notificações
    const container = document.getElementById('chat-notifications-container')
    if (container) {
        container.innerHTML = ''
    }
    
    globalChatInitialized = false
    window.chatGlobalInitialized = false
    
    console.log('✅ Chat Global: Recursos limpos')
}

// ===== HELPER PARA ADICIONAR LISTENERS =====
function addGlobalEventListener(element, event, handler) {
    element.addEventListener(event, handler)
    globalEventListeners.push({ element, event, handler })
}

// ===== HELPER PARA TIMERS =====
function addCleanupTimer(timerId) {
    if (timerId) cleanupTimers.push(timerId)
    return timerId
}

// ===== VERIFICAÇÃO DE DEPENDÊNCIAS =====
function checkDependencies() {
    const missing = []
    
    if (!window.chatGlobalConfig) missing.push('chatGlobalConfig')
    if (!window.fetch) missing.push('fetch API')
    
    if (missing.length > 0) {
        console.warn('⚠️ Chat Global: Dependências faltando:', missing)
        return false
    }
    
    return true
}

// ===== RETRY WRAPPER =====
async function withRetry(fn, maxRetries = 3, delay = 1000) {
    for (let i = 0; i < maxRetries; i++) {
        try {
            return await fn()
        } catch (error) {
            console.warn(`⚠️ Tentativa ${i + 1}/${maxRetries} falhou:`, error.message)
            if (i === maxRetries - 1) throw error
            await new Promise(resolve => setTimeout(resolve, delay * (i + 1)))
        }
    }
}

// ===== INICIALIZAÇÃO CORRIGIDA =====
async function initGlobalChatSystem() {
    if (globalChatInitialized) return true

    try {
        console.log('🌐 Chat Global: Iniciando sistema...')
        
        cleanupGlobalChat()
        loadNotificationConfig()

        if (!checkDependencies()) {
            console.error('❌ Chat Global: Dependências não atendidas')
            return false
        }

        if (!window.chatGlobalConfig?.enabled) {
            console.log('⚠️ Chat Global: Desabilitado')
            return false
        }

        // ✅ SOLICITAR PERMISSÃO DE NOTIFICAÇÃO IMEDIATAMENTE
        await requestNotificationPermission()

        await loadChatServices()
        
        const success = await websocketService.init({
            url: window.chatGlobalConfig.websocket_url,
            debug: window.chatGlobalConfig.debug || false
        })

        if (!success) {
            throw new Error('Falha ao conectar WebSocket')
        }

        setupGlobalListeners()
        await loadInitialChatData()
        integrateWithExistingNotifications()
        startHealthMonitoring()

        globalChatInitialized = true
        window.chatGlobalInitialized = true
        
        console.log('✅ Chat Global: Sistema inicializado')
        
        // ✅ TESTE DE NOTIFICAÇÃO INICIAL
        showToastNotification('Chat Global ativado', 'success', 3000)
        
        return success

    } catch (error) {
        console.error('❌ Chat Global: Erro na inicialização:', error)
        metrics.errors++
        cleanupGlobalChat()
        return false
    }
}

// ===== CARREGAR SERVIÇOS =====
async function loadChatServices() {
    try {
        const { default: websocketServiceModule } = await withRetry(() => 
            import('./servicos/websocket.js')
        )
        websocketService = websocketServiceModule
        window.websocketService = websocketService

        const { default: chatStoreModule } = await withRetry(() => 
            import('./servicos/chatStore.js')
        )
        chatStore = chatStoreModule
        window.chatStore = chatStore

        console.log('✅ Chat Global: Serviços carregados')
    } catch (error) {
        console.error('❌ Chat Global: Erro ao carregar serviços:', error)
        throw error
    }
}

// ===== LISTENERS GLOBAIS CORRIGIDOS =====
function setupGlobalListeners() {
    addGlobalEventListener(window, 'chat-message-received', function(event) {
        const data = event.detail
        metrics.messagesReceived++
        
        console.log('📨 Chat Global: Mensagem recebida:', data)
        
        if (data.message && data.conversation_id) {
            const currentUserId = websocketService.getCurrentUser()?.id
            const isOwnMessage = currentUserId && parseInt(data.message.sender_id) === parseInt(currentUserId)
            
            chatStore.updateConversationList(data.conversation_id, data.message, isOwnMessage)
            chatStore.clearTypingIndicator(data.conversation_id)
            
            debounceSort()
            
            if (!isOwnMessage && !isOnActiveConversation(data.conversation_id)) {
                showGlobalNotification({
                    title: data.message.sender_name || 'Nova Mensagem',
                    message: data.message.content,
                    conversation_id: data.conversation_id,
                    type: 'message'
                })
                playNotificationSound()
            }
            
            updateNotificationBadge()
        }
    })

    // ✅ LISTENER PARA NOTIFICAÇÕES GLOBAIS
    addGlobalEventListener(window, 'chat-global-notification', function(event) {
        const data = event.detail
        
        console.log('📢 Chat Global: Notificação global:', data)
        
        if (data.conversation_id && data.message_preview) {
            const currentUserId = websocketService.getCurrentUser()?.id
            const isOwnMessage = currentUserId && parseInt(data.message_preview.sender_id) === parseInt(currentUserId)
            
            if (!isOwnMessage && !isOnActiveConversation(data.conversation_id)) {
                showGlobalNotification({
                    title: data.message_preview.sender_name || 'Nova Mensagem',
                    message: data.message_preview.content || 'Nova mensagem',
                    conversation_id: data.conversation_id,
                    type: data.type || 'message'
                })
                playNotificationSound()
            }
        }
    })

    addGlobalEventListener(window, 'chat-user-typing', function(event) {
        const data = event.detail
        
        if (data.conversation_id && data.user_id) {
            const currentUserId = websocketService.getCurrentUser()?.id
            if (currentUserId && parseInt(data.user_id) === parseInt(currentUserId)) {
                return
            }

            chatStore.updateTypingIndicator(
                data.conversation_id,
                data.user_id,
                data.user_name || 'Usuário',
                data.is_typing
            )
            
            if (data.is_typing) {
                debounceSort()
            }
        }
    })

    addGlobalEventListener(window, 'chat-conversation-read', function(event) {
        const data = event.detail
        if (data.conversation_id) {
            chatStore.markConversationAsReadGlobal(data.conversation_id)
            updateNotificationBadge()
        }
    })

    addGlobalEventListener(window, 'websocket-connected', function() {
        console.log('🔗 Chat Global: WebSocket conectado')
        showToastNotification('Chat conectado', 'success', 2000)
        websocketService?.joinRoom('general')
        metrics.reconnections++
    })

    addGlobalEventListener(window, 'websocket-disconnected', function() {
        console.log('🚪 Chat Global: WebSocket desconectado')
        showToastNotification('Chat desconectado', 'warning', 3000)
    })

    addGlobalEventListener(window, 'beforeunload', cleanupGlobalChat)
    addGlobalEventListener(window, 'pagehide', cleanupGlobalChat)
}

// ===== DEBOUNCE PARA SORT =====
function debounceSort() {
    if (sortTimeout) clearTimeout(sortTimeout)
    sortTimeout = addCleanupTimer(setTimeout(() => {
        chatStore.sortConversations()
        sortTimeout = null
    }, 150))
}

// ===== UTILITÁRIOS =====
function isOnActiveConversation(conversationId) {
    const isOnChatPage = window.location.pathname.includes('/chat')
    const isConversationActive = chatStore?.conversaSelecionada?.value?.id === parseInt(conversationId)
    return isOnChatPage && isConversationActive
}

async function loadInitialChatData() {
    try {
        if (window.Innoclapps?.request) {
            const response = await window.Innoclapps.request('/chat/conversations/summary')
            if (response.data?.unread_count) {
                updateNotificationBadge(response.data.unread_count)
            }
        }
    } catch (error) {
        console.warn('⚠️ Chat Global: Erro ao carregar dados iniciais:', error)
    }
}

// ===== CONFIGURAÇÃO DE NOTIFICAÇÕES =====
function loadNotificationConfig() {
    try {
        const saved = localStorage.getItem('chat-notification-config')
        if (saved) {
            Object.assign(notificationConfig, JSON.parse(saved))
        }
    } catch (error) {
        console.warn('⚠️ Erro ao carregar configurações de notificação')
    }
}

function configureNotifications(config = {}) {
    Object.assign(notificationConfig, config)
    
    try {
        localStorage.setItem('chat-notification-config', JSON.stringify(notificationConfig))
    } catch (error) {
        console.warn('⚠️ Não foi possível salvar configurações de notificação')
    }
}

// ✅ SOLICITAR PERMISSÃO DE NOTIFICAÇÃO
async function requestNotificationPermission() {
    if (!('Notification' in window)) {
        console.warn('⚠️ Browser não suporta notificações')
        return false
    }

    if (Notification.permission === 'granted') {
        console.log('✅ Permissão de notificação já concedida')
        return true
    }

    if (Notification.permission === 'denied') {
        console.warn('⚠️ Permissão de notificação negada')
        return false
    }

    try {
        const permission = await Notification.requestPermission()
        console.log('🔔 Permissão de notificação:', permission)
        return permission === 'granted'
    } catch (error) {
        console.error('❌ Erro ao solicitar permissão:', error)
        return false
    }
}

// ===== NOTIFICAÇÕES OTIMIZADAS =====
function showGlobalNotification({ title, message, conversation_id, type = 'message' }) {
    if (!notificationConfig.enabled) return

    try {
        metrics.notificationsShown++

        // ✅ NOTIFICAÇÃO DO BROWSER
        if (notificationConfig.desktop && 'Notification' in window && Notification.permission === 'granted') {
            const notification = new Notification(title, {
                body: truncateMessage(message),
                icon: '/favicon.ico',
                tag: `chat-${conversation_id}`,
                requireInteraction: false,
                silent: !notificationConfig.sound
            })

            notification.onclick = function() {
                window.focus()
                openChat(conversation_id)
                notification.close()
            }

            addCleanupTimer(setTimeout(() => notification.close(), notificationConfig.duration))
        }

        // ✅ TOAST NOTIFICATION
        showToastNotification(`${title}: ${truncateMessage(message)}`, getToastType(type), notificationConfig.duration)

        console.log('🔔 Notificação mostrada:', { title, conversation_id, type })

    } catch (error) {
        console.error('❌ Chat Global: Erro ao mostrar notificação:', error)
        metrics.errors++
    }
}

function truncateMessage(message) {
    if (!message || typeof message !== 'string') return 'Nova mensagem'
    return message.length > 100 ? message.substring(0, 97) + '...' : message
}

function getToastType(type) {
    switch (type) {
        case 'whatsapp_message':
        case 'internal_message':
        case 'message':
            return 'info'
        case 'error':
            return 'error'
        case 'warning':
            return 'warning'
        default:
            return 'info'
    }
}

function showToastNotification(message, type = 'info', duration = 4000) {
    const now = Date.now()
    
    // ✅ THROTTLE
    if (lastNotification.type === type && (now - lastNotification.time) < notificationConfig.throttleTime) {
        return
    }
    
    lastNotification = { type, time: now }

    try {
        let container = document.getElementById('chat-notifications-container')
        if (!container) {
            container = document.createElement('div')
            container.id = 'chat-notifications-container'
            container.className = 'chat-notifications-container'
            document.body.appendChild(container)
        }

        // ✅ LIMITAR NÚMERO DE NOTIFICAÇÕES
        const existingToasts = container.children
        if (existingToasts.length >= notificationConfig.maxVisible) {
            existingToasts[0].remove()
        }

        const toast = createToastElement(message, type)
        container.appendChild(toast)

        // ✅ ANIMAÇÃO DE ENTRADA
        addCleanupTimer(setTimeout(() => {
            toast.classList.add('chat-toast-show')
        }, 100))

        // ✅ AUTO CLOSE
        const autoCloseTimer = addCleanupTimer(setTimeout(() => {
            closeToast(toast)
        }, duration))

        // ✅ CLOSE BUTTON
        const closeBtn = toast.querySelector('.chat-toast-close')
        if (closeBtn) {
            closeBtn.onclick = () => {
                clearTimeout(autoCloseTimer)
                closeToast(toast)
            }
        }

        console.log('🍞 Toast mostrado:', { message, type, duration })

    } catch (error) {
        console.error('❌ Chat Global: Erro ao mostrar toast:', error)
        metrics.errors++
    }
}

function createToastElement(message, type) {
    const toast = document.createElement('div')
    toast.className = `chat-toast chat-toast-${type}`

    const parts = message.split(': ')
    const hasTitle = parts.length > 1

    toast.innerHTML = `
        <div class="chat-toast-content">
            <div class="chat-toast-message">
                ${hasTitle ? `<div class="chat-toast-title">${parts[0]}</div>` : ''}
                <div>${hasTitle ? parts.slice(1).join(': ') : message}</div>
            </div>
            <button class="chat-toast-close" type="button">&times;</button>
        </div>
    `

    return toast
}

function closeToast(toast) {
    if (!toast || !toast.parentNode) return

    toast.classList.remove('chat-toast-show')
    addCleanupTimer(setTimeout(() => {
        if (toast.parentNode) {
            toast.remove()
        }
    }, 300))
}

function updateNotificationBadge(count = null) {
    try {
        if (count === null) {
            count = chatStore?.getStats()?.naoLidas || 0
        }

        // ✅ ATUALIZAR BADGES EXISTENTES
        const badges = document.querySelectorAll('.chat-notification-badge')
        badges.forEach(badge => {
            if (count > 0) {
                badge.textContent = count > 99 ? '99+' : count
                badge.classList.add('show')
            } else {
                badge.classList.remove('show')
            }
        })

        // ✅ ATUALIZAR TÍTULO DA PÁGINA
        const originalTitle = document.title.replace(/^\(\d+\)\s/, '')
        document.title = count > 0 ? `(${count}) ${originalTitle}` : originalTitle

        // ✅ BADGE DO NAVEGADOR (SE SUPORTADO)
        if ('setAppBadge' in navigator) {
            if (count > 0) {
                navigator.setAppBadge(count).catch(() => {})
            } else {
                navigator.clearAppBadge().catch(() => {})
            }
        }

    } catch (error) {
        console.error('❌ Chat Global: Erro ao atualizar badge:', error)
    }
}

function playNotificationSound() {
    if (!notificationConfig.sound) return

    try {
        // ✅ SOM SIMPLES GERADO PROGRAMATICAMENTE
        const audioContext = new (window.AudioContext || window.webkitAudioContext)()
        const oscillator = audioContext.createOscillator()
        const gainNode = audioContext.createGain()

        oscillator.connect(gainNode)
        gainNode.connect(audioContext.destination)

        oscillator.frequency.setValueAtTime(800, audioContext.currentTime)
        oscillator.frequency.setValueAtTime(600, audioContext.currentTime + 0.1)

        gainNode.gain.setValueAtTime(0.1, audioContext.currentTime)
        gainNode.gain.exponentialRampToValueAtTime(0.01, audioContext.currentTime + 0.2)

        oscillator.start(audioContext.currentTime)
        oscillator.stop(audioContext.currentTime + 0.2)

    } catch (error) {
        // ✅ FALLBACK PARA AUDIO ELEMENT
        try {
            const audio = new Audio()
            audio.src = 'data:audio/wav;base64,UklGRnoGAABXQVZFZm10IBAAAAABAAEAQB8AAEAfAAABAAgAZGF0YQoGAACBhYqFbF1fdJivrJBhNjVgodDbq2EcBj+a2/LDciUFLIHO8tiJNwgZaLvt559NEAxQp+PwtmMcBjiR1/LMeSwFJHfH8N2QQAoUXrTp66hVFApGn+DyvmMeAjuZ2e/AcSEELYPP8dWLOAcTZLnp6Z1NEAxOpd/yxnkpBSl+zPLaizsIGGS57OihUgwLUarm7K5fFgU9k9n0unAhBC2Iz/DUizgHEWG56+OZURE='
            audio.volume = 0.3
            audio.play().catch(() => {})
        } catch (fallbackError) {
            // Ignorar se não conseguir reproduzir som
        }
    }
}

function integrateWithExistingNotifications() {
    try {
        // ✅ PROCURAR BOTÕES DE NOTIFICAÇÃO EXISTENTES
        const selectors = [
            '[data-notifications-button]',
            '.notifications-button',
            '[data-testid="notifications"]',
            '.notification-bell',
            '.nav-notifications'
        ]

        for (const selector of selectors) {
            const button = document.querySelector(selector)
            if (button && !button.querySelector('.chat-notification-badge')) {
                const badge = document.createElement('span')
                badge.className = 'chat-notification-badge'
                
                if (button.style.position !== 'relative') {
                    button.style.position = 'relative'
                }
                
                button.appendChild(badge)
                console.log('✅ Badge integrado ao botão:', selector)
                break
            }
        }

    } catch (error) {
        console.warn('⚠️ Chat Global: Erro ao integrar notificações:', error)
    }
}

// ===== HEALTH MONITORING =====
function performHealthCheck() {
    const issues = []
    
    if (!websocketService?.isConnected) {
        issues.push('WebSocket desconectado')
    }
    
    if (!chatStore) {
        issues.push('ChatStore não inicializado')
    }
    
    if (globalEventListeners.length === 0) {
        issues.push('Nenhum listener ativo')
    }
    
    if (issues.length > 0) {
        console.warn('⚠️ Chat Global Health Check:', issues)
        return false
    }
    
    return true
}

function startHealthMonitoring() {
    const healthTimer = setInterval(() => {
        if (!performHealthCheck()) {
            console.log('🔄 Chat Global: Tentando recuperar...')
            if (websocketService && !websocketService.isConnected) {
                websocketService.restart().catch(console.error)
            }
        }
    }, 30000)
    
    addCleanupTimer(healthTimer)
}

function incrementMetric(name) {
    if (metrics.hasOwnProperty(name)) {
        metrics[name]++
    }
}

function getMetrics() {
    return {
        ...metrics,
        uptime: Date.now() - metrics.startTime,
        avgMessagesPerMinute: metrics.messagesReceived / ((Date.now() - metrics.startTime) / 60000)
    }
}

// ===== FUNÇÕES PÚBLICAS =====
function getChatGlobalStatus() {
    return {
        initialized: globalChatInitialized,
        websocket: websocketService?.getStatus(),
        chatStore: chatStore?.getStats(),
        notifications: notificationConfig,
        metrics: getMetrics(),
        timers: cleanupTimers.length,
        listeners: globalEventListeners.length
    }
}

function openChat(conversationId = null) {
    const url = conversationId ? `/chat?conversation=${conversationId}` : '/chat'
    if (window.location.pathname === '/chat' && conversationId) {
        // Se já estiver no chat, apenas selecionar a conversa
        if (window.chatStore) {
            window.chatStore.selectConversation(conversationId)
        }
    } else {
        window.location.href = url
    }
}

// ===== EXPOSIÇÃO GLOBAL =====
window.getChatGlobalStatus = getChatGlobalStatus
window.openChat = openChat
window.cleanupGlobalChat = cleanupGlobalChat
window.configureChatNotifications = configureNotifications
window.getChatMetrics = getMetrics
window.requestNotificationPermission = requestNotificationPermission

// ===== AUTO-INICIALIZAÇÃO MELHORADA =====
if (typeof window !== 'undefined') {
    // ✅ AGUARDAR DOM READY
    if (document.readyState === 'loading') {
        addGlobalEventListener(document, 'DOMContentLoaded', () => {
            addCleanupTimer(setTimeout(async () => {
                if (window.chatGlobalConfig?.enabled && !window.chatGlobalInitialized) {
                    await initGlobalChatSystem()
                }
            }, 1000)) // ✅ AUMENTAR DELAY PARA 1 SEGUNDO
        })
    } else {
        // ✅ DOM JÁ ESTÁ PRONTO
        addCleanupTimer(setTimeout(async () => {
            if (window.chatGlobalConfig?.enabled && !window.chatGlobalInitialized) {
                await initGlobalChatSystem()
            }
        }, 1000))
    }
}

export default { 
    initGlobalChatSystem, 
    getChatGlobalStatus, 
    openChat, 
    cleanupGlobalChat,
    configureNotifications,
    requestNotificationPermission
}